varying vec2 f_texcoord_depth;
varying vec2 f_texcoord_noise;

uniform sampler2D texture_depth;
uniform sampler2D texture_noise;

uniform mat4 mat_proj;
uniform vec3 view_inv_y;
uniform vec2 tan_half_fov;


const int sampleSphereNum = 8;
const vec3 sampleSphere0 = vec3( 0.349436,  0.165796,  0.385136);
const vec3 sampleSphere1 = vec3(-0.361479,  0.398428,  0.328603);
const vec3 sampleSphere2 = vec3(-0.466405,  0.305378,  0.492064);
const vec3 sampleSphere3 = vec3(-0.166569, -0.172616,  0.143073);
const vec3 sampleSphere4 = vec3( 0.204571,  0.375978,  0.834151);
const vec3 sampleSphere5 = vec3( 0.127111, -0.169984,  0.431604);
const vec3 sampleSphere6 = vec3( 0.089099, -0.269580,  0.469807);
const vec3 sampleSphere7 = vec3( 0.048708, -0.001401,  0.032761);

#define SSAO_STRENGTH_ADAPT_MULT 2.5


vec3 getPositionView(in vec2 texcoord, in float depth)
{
	vec2 ndc = texcoord * 2.0 - 1.0;
	vec3 viewRay = vec3(ndc.x * tan_half_fov.x, ndc.y * tan_half_fov.y, -1.0);
	vec3 pos = viewRay * depth;
	return pos;
}

vec3 getNormalFromDepth(in float depth, in vec3 posView)
{
	float offsetRange = 0.0002 * (1.0-depth);
	vec2 offset1 = f_texcoord_depth + vec2(0.0, offsetRange);
	vec2 offset2 = f_texcoord_depth + vec2(offsetRange, 0.0);

	float depth1 = texture2D(texture_depth, offset1).r;
	float depth2 = texture2D(texture_depth, offset2).r;

	vec3 p1 = getPositionView(offset1, depth1);
	vec3 p2 = getPositionView(offset2, depth2);

	vec3 v1 = posView-p1;
	vec3 v2 = posView-p2;

	vec3 normal = normalize(cross(v2, v1));
	return normal;
}

float calcOcclusion(in vec3 sample, in mat3 tbn, in vec3 posView)
{
	const float radius = 0.0003;
	const float range = 2.0*radius;

	vec3 rayHemi = tbn * sample;
	rayHemi = posView + rayHemi * radius;

	vec4 offset = vec4(rayHemi, 1.0);
	offset = mat_proj * offset;
	offset.xy /= offset.w;
	offset.xy = offset.xy * 0.5 + 0.5;

	float depth = texture2D(texture_depth, offset.xy).r;

	float factorRange = 1.0 - min(abs(-posView.z - depth) / range, 1.0);
	float factorOcclusion = step(depth, -rayHemi.z);
	return factorRange * factorOcclusion;
}

void main(void)
{
	const float ignoreThresholdDepth = 0.1;
	float depth = texture2D(texture_depth, f_texcoord_depth).r;
	if(depth > ignoreThresholdDepth)
	{
		gl_FragColor = vec4(0.0, 0.0, 0.0, 1.0);
		return;
	}

	vec3 posView = getPositionView(f_texcoord_depth, depth);
	vec3 normal = getNormalFromDepth(depth, posView);

	vec3 rnd = texture2D(texture_noise, f_texcoord_noise).rgb * 2.0 - vec3(1.0);

	vec3 tangent = normalize(rnd - normal * dot(rnd, normal));
	vec3 bitangent = cross(normal, tangent);
	mat3 tbn = mat3(tangent, bitangent, normal);

	float occlusion = 0.0;
	occlusion += calcOcclusion(sampleSphere0, tbn, posView);
	occlusion += calcOcclusion(sampleSphere1, tbn, posView);
	occlusion += calcOcclusion(sampleSphere2, tbn, posView);
	occlusion += calcOcclusion(sampleSphere3, tbn, posView);
	occlusion += calcOcclusion(sampleSphere4, tbn, posView);
	occlusion += calcOcclusion(sampleSphere5, tbn, posView);
	occlusion += calcOcclusion(sampleSphere6, tbn, posView);
	occlusion += calcOcclusion(sampleSphere7, tbn, posView);
	occlusion /= float(sampleSphereNum);

	float normalYworld = dot(normal, view_inv_y);
	const float normalYthreshold = -0.4;
	occlusion *= step(normalYthreshold, normalYworld);

	occlusion = min(1.0, occlusion * SSAO_STRENGTH_ADAPT_MULT);

	gl_FragColor = vec4(occlusion, occlusion, occlusion, 1.0);
}
